import { useState, useEffect } from 'react'
import { CheckCircle2, XCircle, Calendar, Clock, User, Star } from 'lucide-react'
import ImageSlider from '../components/ImageSlider'
import './MyBookings.css'

const MyBookings = () => {
  const [activeTab, setActiveTab] = useState('active')
  const [bookings, setBookings] = useState({
    active: [],
    past: []
  })

  useEffect(() => {
    const savedBookings = localStorage.getItem('fitnessBookings')
    if (savedBookings) {
      const parsed = JSON.parse(savedBookings)
      setBookings(parsed)
    } else {
      const defaultBookings = {
        active: [
          {
            id: 1,
            type: 'Йога',
            date: '2025-03-20',
            time: '18:00',
            trainer: 'Анна Петрова',
            status: 'active'
          },
          {
            id: 2,
            type: 'Кроссфит',
            date: '2025-03-22',
            time: '19:30',
            trainer: 'Иван Сидоров',
            status: 'active'
          },
          {
            id: 3,
            type: 'Пилатес',
            date: '2025-03-24',
            time: '17:00',
            trainer: 'Мария Козлова',
            status: 'active'
          }
        ],
        past: [
          {
            id: 4,
            type: 'Йога',
            date: '2025-03-15',
            time: '18:00',
            trainer: 'Анна Петрова',
            rating: 5
          },
          {
            id: 5,
            type: 'Стретчинг',
            date: '2025-03-10',
            time: '20:00',
            trainer: 'Ольга Новикова',
            rating: 4
          },
          {
            id: 6,
            type: 'Кроссфит',
            date: '2025-03-08',
            time: '19:30',
            trainer: 'Иван Сидоров',
            rating: 5
          }
        ]
      }
      setBookings(defaultBookings)
      localStorage.setItem('fitnessBookings', JSON.stringify(defaultBookings))
    }

    const handleStorageChange = () => {
      const updated = localStorage.getItem('fitnessBookings')
      if (updated) {
        setBookings(JSON.parse(updated))
      }
    }

    window.addEventListener('storage', handleStorageChange)
    const interval = setInterval(handleStorageChange, 500)

    return () => {
      window.removeEventListener('storage', handleStorageChange)
      clearInterval(interval)
    }
  }, [])

  const sliderImages = [
    '/images/fitness-1.jpg',
    '/images/fitness-2.jpg',
    '/images/fitness-3.jpg',
    '/images/fitness-4.jpg'
  ]

  const handleRating = (bookingId, rating) => {
    const updatedBookings = {
      ...bookings,
      past: bookings.past.map(booking => 
        booking.id === bookingId ? { ...booking, rating } : booking
      )
    }
    setBookings(updatedBookings)
    localStorage.setItem('fitnessBookings', JSON.stringify(updatedBookings))
  }

  const features = [
    { icon: '/icons/stat-activity.svg', title: 'Интенсивные тренировки', color: '#e63946' },
    { icon: '/icons/stat-target.svg', title: 'Достижение целей', color: '#457b9d' },
    { icon: '/icons/stat-zap.svg', title: 'Быстрый результат', color: '#f77f00' },
    { icon: '/icons/stat-shield.svg', title: 'Безопасность', color: '#06d6a0' }
  ]

  return (
    <div className="bookings-page">
      <div className="welcome-banner">
        <div className="header-shapes">
          <div className="shape shape-1"></div>
          <div className="shape shape-2"></div>
          <div className="shape shape-3"></div>
          <div className="shape shape-4"></div>
          <div className="shape shape-5"></div>
        </div>
        <h1 className="welcome-title">FitZone</h1>
        <p className="welcome-text">Превратите мечты о здоровом теле в реальность</p>
      </div>

      <div className="features-section">
        {features.map((feature, index) => (
          <div key={index} className="feature-box" style={{ '--feature-color': feature.color }}>
            <img src={feature.icon} alt={feature.title} className="feature-icon" width="32" height="32" />
            <span className="feature-title">{feature.title}</span>
          </div>
        ))}
      </div>

      <ImageSlider images={sliderImages} />
      
      <div className="tabs-container">
        <button
          className={`tab-btn ${activeTab === 'active' ? 'active' : ''}`}
          onClick={() => setActiveTab('active')}
        >
          <CheckCircle2 className="tab-icon" size={20} />
          Активные
        </button>
        <button
          className={`tab-btn ${activeTab === 'past' ? 'active' : ''}`}
          onClick={() => setActiveTab('past')}
        >
          <XCircle className="tab-icon" size={20} />
          Прошедшие
        </button>
      </div>

      <div className="bookings-list">
        {activeTab === 'active' ? (
          bookings.active.length > 0 ? (
            bookings.active.map((booking) => (
              <div key={booking.id} className="booking-card active-card">
                <div className="card-header">
                  <div className="type-badge active-badge">{booking.type}</div>
                  <span className="status-dot active-dot"></span>
                </div>
                <div className="card-body">
                  <div className="info-row">
                    <Calendar className="info-icon" size={20} />
                    <span>{new Date(booking.date).toLocaleDateString('ru-RU', { day: 'numeric', month: 'long', year: 'numeric' })}</span>
                  </div>
                  <div className="info-row">
                    <Clock className="info-icon" size={20} />
                    <span>{booking.time}</span>
                  </div>
                  <div className="info-row">
                    <User className="info-icon" size={20} />
                    <span>{booking.trainer}</span>
                  </div>
                </div>
              </div>
            ))
          ) : (
            <div className="empty-state">
              <p>Нет активных записей</p>
            </div>
          )
        ) : (
          bookings.past.length > 0 ? (
            bookings.past.map((booking) => (
              <div key={booking.id} className="booking-card past-card">
                <div className="card-header">
                  <div className="type-badge past-badge">{booking.type}</div>
                  <span className="status-dot past-dot"></span>
                </div>
                <div className="card-body">
                  <div className="info-row">
                    <Calendar className="info-icon" size={20} />
                    <span>{new Date(booking.date).toLocaleDateString('ru-RU', { day: 'numeric', month: 'long', year: 'numeric' })}</span>
                  </div>
                  <div className="info-row">
                    <Clock className="info-icon" size={20} />
                    <span>{booking.time}</span>
                  </div>
                  <div className="info-row">
                    <User className="info-icon" size={20} />
                    <span>{booking.trainer}</span>
                  </div>
                </div>
                <div className="rating-section">
                  <p className="rating-title">Оценка тренера</p>
                  <div className="stars-container">
                    {[1, 2, 3, 4, 5].map((star) => (
                      <button
                        key={star}
                        className={`star-button ${booking.rating >= star ? 'filled' : ''}`}
                        onClick={() => handleRating(booking.id, star)}
                      >
                        <Star className="star-icon" size={28} fill={booking.rating >= star ? 'currentColor' : 'none'} />
                      </button>
                    ))}
                  </div>
                </div>
              </div>
            ))
          ) : (
            <div className="empty-state">
              <p>Нет прошедших записей</p>
            </div>
          )
        )}
      </div>
    </div>
  )
}

export default MyBookings

